analyzeVideo2(videoFileName, labels)
% This function takes a grayscale video as input and produces a graph of
% how an ROI's intensity changes over time.  After providing the input
% video, the user is asked to select one background ROI and one or more
% foreground ROIs for analysis from any frame of the video.

% The program works as follows:
% (1) Ask the user for ROIs, with a magnified, scrollable, flippable heat
% map.  Note that the ROI's size will never change, but it may be
% translated in space depending on what happens to its underlying objects.
% (2) Identify the objects that make up the ROI
% (3) For each frame before and after the frame on which the ROI was drawn,
%    follow the objects that were in the ROI (based on spatial overlap).  
%    Several things can happen to the objects that will automatically be 
%    accounted for by the ROI:
%    (a) The object translates in space.  If this happens, translate the ROI
%    with the same translation vector.  The vector is determined by the
%    movement of the object's centroid from one frame to the next.
%    (b) The object scales in space (increases or decreases).  There are 2
%    kinds of scaling:
%        (i) Scaling that maintains the object's centroid location.  This is
%        a uniform expansion or contraction, and in this case the ROI is
%        not moved.
%        (ii) Scaling that shifts the object's centroid location.  This
%        happens if only part of the object expands or contracts.  In this
%        case, the ROI will be translated by the object's centroid vector
%        shift.
%    (c) The object collides with another object.  In this case, create
%    artifical object boundaries by using the object boundaries from the
%    previous frame where the objects were not touching.  Deal with the ROI
%    as above.
%    (d) The object separates into two or more objects.  Take the centroid
%    of both objects together and use that for shifting the ROI.

% This function can be run on its own or as part of the imaging recording
% package, which also controls the Polychrome light source and Andor EM
% camera.

global vreader;

% PARAMETERS
lw = 2;  % LineWidth

tic;

vreader = VideoReader(videoFileName);
[p vidName ext] = fileparts(videoFileName);
fr = vreader.FrameRate;
numFrames = vreader.NumberOfFrames;
numROI = length(labels);
vidWidth = vreader.Width;
vidHeight = vreader.Height;

% Start up the GUI
h = layout_manual(vreader, numROI);
waitfor(h, 'Pointer', 'watch');

handles = guidata(h);

fluorescence = zeros(numFrames, numROI);
fluorescenceBG = zeros(numFrames, 1);

for i = 1:numFrames
    curFrame = read(vreader, i);
    for j = 1:numROI
        mask = poly2mask(handles.roiPos{i,j}(:,1), handles.roiPos{i,j}(:,2), vidHeight, vidWidth);
        fluorescence(i, j) = sum(curFrame(:,:,1) .* uint8(mask)) / sum(mask);        
    end
    bgmask = poly2mask(handles.roiPos{i, numROI+1}(:,1), handles.roiPos{i,numROI+1}(:,2), vidHeight, vidWidth);
    fluorescenceBG(i) = sum(curFrame(:,:,1) .* uint8(bgmask)) / sum(bgmask);
end

rawfluorescence = fluorescence;
% Do background subtraction!
for i = 1:numROI
    fluorescence(:,i) = (fluorescence(:,i) - fluorescence(1,i)) ./ ...
                        (fluorescence(1, i) - fluorescenceBG);
end

f = figure;
plot((1:numFrames)/fr, fluorescence*100, 'LineWidth', lw);
ylabel('\DeltaF/F (%)', 'FontWeight', 'bold');
xlabel('sec', 'FontWeight', 'bold');
title(vidName, 'FontWeight', 'bold');
prettify;
legend(labels);

roiPos = handles.roiPos;

eval([vidName ' = fluorescence;']);
if (exist([vidName '.mat'], 'file') == 0)  % Don't overwrite if there's an existing file
    save([vidName, '.mat'], vidName, 'rawfluorescence', 'fluorescenceBG', 'roiPos');
end
saveas(f, [vidName '.fig']);


toc;
beep;
